#include <glut.h>

/*
 * pbrt source code Copyright(c) 1998-2005 Matt Pharr and Greg Humphreys
 *
 * All Rights Reserved.
 * For educational use only; commercial use expressly forbidden.
 * NO WARRANTY, express or implied, for this software.
 * (See file License.txt for complete license)
 */

// trianglemesh.cpp*
#include "pbrt.h"
#include "shape.h"
#include "paramset.h"

#include "trianglemesh.h"
#include "scene.h"

static void print3f(const void *d)
{
	const float *f = (const float *)d;
	printf("%f %f %f\n",f[0], f[1], f[2]);
}

#ifdef __CODEDEFT__
#define FORCE_COEFF 2.f
void PrevInsert(IntList** prev, int i)
{
	IntList* newlist = new IntList();
	newlist->i = i;
	newlist->next = *prev;
	*prev = newlist;
}
bool PrevNotFound(IntList* prev, int i)
{
	IntList* cur = prev;
	while (cur != 0) {
		if (cur->i == i)
			return false;
		cur = cur->next;
	}
	return true;
}
// Return all the neighbor triangles' neighbor points
void NeighborTri::TraverseNeighbors(NeighSet& neighs, int numrecurses, IntList** prevTraversed, int id)
{
	if (numrecurses == 0)
		return;

	PointList* cur = 0;
	if (plist1 != 0 && plist2 == 0) {
		cur = plist1;
	} else if (plist2 != 0 && plist1 == 0) {
		cur = plist2;
	} else if (plist1 == 0 && plist2 == 0) {
		cur = 0;
	} else {
		Assert(0);
	}

	while (cur != 0) {
		neighs.insert(NeighborPoint(cur->p, id));
		cur = cur->next;
	}

	PrevInsert(prevTraversed, myId);
	if (n1 != -1 && PrevNotFound(*prevTraversed, n1)) {
		neighbor1->TraverseNeighbors(neighs, numrecurses-1, prevTraversed, id);
	}
	if (n2 != -1 && PrevNotFound(*prevTraversed, n2)) {
		neighbor2->TraverseNeighbors(neighs, numrecurses-1, prevTraversed, id);
	}
	if (n3 != -1 && PrevNotFound(*prevTraversed, n3)) {
		neighbor3->TraverseNeighbors(neighs, numrecurses-1, prevTraversed, id);
	}
}

void NeighborTri::RenderGL(Point& p1, Point& p2, Point& p3, int nverts)
{
	glPointSize(5);
	glBegin(GL_POINTS);
	float c1 = ((float)((myId + 1) % nverts)) / (float)nverts;
	float c2 = ((float)((myId + 2007) % nverts)) / (float)nverts;
	float c3 = ((float)((myId + 3007) % nverts)) / (float)nverts;
//	float c1 = 1.f, c2 = 1.f, c3 = 1.f;
	glColor3f(c1, c2, c3);
	MoveablePoint* cur = mps;
	for (int i = 0; i < numMoveables; i++) {
		Point p(cur->b1*p1 + cur->b2*p2 + (1.f-cur->b1-cur->b2)*p3);
		glVertex3f(p.x, p.y, p.z);
		cur = cur->next;
	}
	glEnd();

	/*
	float c1, c2, c3;
	Point mid = p1*0.33f + p2*0.33f + p3*0.34f;
	glBegin(GL_TRIANGLES);

	c1 = ((float)((3*(n1+1) * 3*(myId+1))%nverts))/((float)nverts);
	c2 = ((float)((7*(n1+1) * 7*(myId+1))%nverts))/((float)nverts);
	c3 = ((float)((13*(n1+1) * 13*(myId+1))%nverts))/((float)nverts);
	glColor3f(c1, c2, c3);
	glVertex3f(p1.x, p1.y, p1.z);
	glVertex3f(p2.x, p2.y, p2.z);
	glVertex3f(mid.x, mid.y, mid.z);

	c1 = ((float)((3*(n2+1) * 3*(myId+1))%nverts))/((float)nverts);
	c2 = ((float)((7*(n2+1) * 7*(myId+1))%nverts))/((float)nverts);
	c3 = ((float)((13*(n2+1) * 13*(myId+1))%nverts))/((float)nverts);
	glColor3f(c1, c2, c3);
	glVertex3f(p2.x, p2.y, p2.z);
	glVertex3f(p3.x, p3.y, p3.z);
	glVertex3f(mid.x, mid.y, mid.z);

	c1 = ((float)((3*(n3+1) * 3*(myId+1))%nverts))/((float)nverts);
	c2 = ((float)((7*(n3+1) * 7*(myId+1))%nverts))/((float)nverts);
	c3 = ((float)((13*(n3+1) * 13*(myId+1))%nverts))/((float)nverts);
	glColor3f(c1, c2, c3);
	glVertex3f(p3.x, p3.y, p3.z);
	glVertex3f(p1.x, p1.y, p1.z);
	glVertex3f(mid.x, mid.y, mid.z);

	glEnd();
	*/
}
#endif

// TriangleMesh Method Definitions
TriangleMesh::TriangleMesh(const Transform &o2w, bool ro,
		int nt, int nv, const int *vi, const Point *P,
		const Normal *N, const Vector *S, const float *uv)
	: Shape(o2w, ro) {
	ntris = nt;
	nverts = nv;
	vertexIndex = new int[3 * ntris];
	memcpy(vertexIndex, vi, 3 * ntris * sizeof(int));
	// Copy _uv_, _N_, and _S_ vertex data, if present
	if (uv) {
		uvs = new float[2*nverts];
		memcpy(uvs, uv, 2*nverts*sizeof(float));
	}
	else uvs = NULL;
	p = new Point[nverts];
	if (N) {
		n = new Normal[nverts];
		memcpy(n, N, nverts*sizeof(Normal));
	}
	else n = NULL;
	if (S) {
		s = new Vector[nverts];
		memcpy(s, S, nverts*sizeof(Vector));
	}
	else s = NULL;
	// Transform mesh vertices to world space
	for (int i  = 0; i < nverts; ++i)
		p[i] = ObjectToWorld(P[i]);
#ifdef __CODEDEFT__
	/*
	// Quick file test
	bSave = false;
	float totalArea_ = 0.f;
	for (int i  = 0; i < ntris; ++i) {
		int v1 = vertexIndex[3*i + 0];
		int v2 = vertexIndex[3*i + 1];
		int v3 = vertexIndex[3*i + 2];

		const Point &p1 = WorldToObject(p[v1]);
		const Point &p2 = WorldToObject(p[v2]);
		const Point &p3 = WorldToObject(p[v3]);

		Point fixedPoint = (1.f/3.f)*p1 + (1.f/3.f)*p2 + (1.f/3.f)*p3;

		// What's the area of this triangle ?
		Vector side1(p2 - p1);
		Vector side2(p3 - p1);
		Vector normside1 = Normalize(side1);
		Vector normside2 = Normalize(side2);

		Vector tmpNormal = Cross(normside1, normside2);
		Normal fixedNormal(tmpNormal);
		fixedNormal = Normalize(fixedNormal);

		fixedPoints.push_back(fixedPoint);
		fixedNormals.push_back(fixedNormal);

		float dotp = Dot(normside1, normside2);
		float sintheta = sinf(acosf(dotp));
//		float area = side1.Length() * side2.Length() * sintheta;
		float area = Cross(side1, side2).Length() * (0.5f);
		if (!(area > 0.f)) {
			//			printf("%f %f %f\n", neightris[i].n.x, neightris[i].n.y, neightris[i].n.z);
			//			printf("Area %f Sintheta %f Dotp %f\n", area, sintheta, dotp);
			area = 0.f;
			Assert(0);
		}

		fixedAreas.push_back(area);

		totalArea_ += area;
	}
	fixedArea = totalArea_;
	return;
	*/












	bFileLoaded = false;
	bValidNeighbors = true;
	seHashTable = new SEList[nverts];
	memset(seHashTable, 0, sizeof(SEList)*nverts);
	neightris = new NeighborTri[ntris];

	triangleAreas = new float[ntris];

	for (int i = 0; i < ntris; ++i) {
		int v1 = vertexIndex[3*i];
		int v2 = vertexIndex[3*i + 1];
		int v3 = vertexIndex[3*i + 2];

		// First edge
		SharedEdge se1(p[v1], p[v2], i);
		SharedEdge se2(p[v2], p[v3], i);
		SharedEdge se3(p[v3], p[v1], i);

		int sn1 = GetSharedNeighbor(se1);
		int sn2 = GetSharedNeighbor(se2);
		int sn3 = GetSharedNeighbor(se3);

		se1.tri2Id = sn1;
		se2.tri2Id = sn2;
		se3.tri2Id = sn3;

		// This is one triangle
		neightris[i].myId = i;
		neightris[i].v1 = v1;
		neightris[i].v2 = v2;
		neightris[i].v3 = v3;
		neightris[i].n1 = sn1;
		if (sn1 != -1) {
			UpdateNeighbors(i,sn1,se1);
			neightris[i].neighbor1 = &neightris[sn1];
		} else {
			neightris[i].e1 = se1;
		}
		neightris[i].n2 = sn2;
		if (sn2 != -1) {
			UpdateNeighbors(i,sn2,se2);
			neightris[i].neighbor2 = &neightris[sn2];
		} else {
			neightris[i].e2 = se2;
		}
		neightris[i].n3 = sn3;
		if (sn3 != -1) {
			UpdateNeighbors(i,sn3,se3);
			neightris[i].neighbor3 = &neightris[sn3];
		} else {
			neightris[i].e3 = se3;
		}
	}
#endif


	// YG add begin
	faceN = new Normal[ntris];
#ifdef __CODEDEFT__
	float totalArea = 0.f;
#endif
	for (int i  = 0; i < ntris; ++i) {
		int v1 = vertexIndex[3*i + 0];
		int v2 = vertexIndex[3*i + 1];
		int v3 = vertexIndex[3*i + 2];

		const Point &p1 = p[v1];
		const Point &p2 = p[v2];
		const Point &p3 = p[v3];

#ifdef __CODEDEFT__
		// What's the area of this triangle ?
		Vector side1(p2 - p1);
		Vector side2(p3 - p1);
		Vector normside1 = Normalize(side1);
		Vector normside2 = Normalize(side2);

		neightris[i].n = Cross(normside1, normside2);
		neightris[i].n = Normalize(neightris[i].n);

		float dotp = Dot(normside1, normside2);
		float sintheta = sinf(acosf(dotp));
		float area = side1.Length() * side2.Length() * sintheta;
		if (!(area > 0.f)) {
//			printf("%f %f %f\n", neightris[i].n.x, neightris[i].n.y, neightris[i].n.z);
//			printf("Area %f Sintheta %f Dotp %f\n", area, sintheta, dotp);
			area = 0.f;
		}
		totalArea += area;
		triangleAreas[i] = totalArea;
		neightris[i].plist1 = 0;
		neightris[i].plist2 = 0;
#endif

		Vector normal = Cross(p2 - p1, p3 - p1);
		if (normal.Length()==0) {
			printf("degenerate normal %d %d %d\n", v1, v2, v3);
			print3f(&p1);
			print3f(&p2);
			print3f(&p3);
			continue;
		}
		else {
			normal = Normalize(normal);
		}
		faceN[i] = (Normal) normal;
	}
	// YG add end

#ifdef __CODEDEFT__
	bSave = false;
	if (!(totalArea > 0.f))
		printf("Total Area %f\n", totalArea);
	// Randomly distribute points across the mesh
//	int numPointsAssigned = (int)totalArea / 200;
	int numPointsAssigned = (int)totalArea / 2;
	for (int i = 0; i < numPointsAssigned; i++) {
		float randomnum = RandomFloat() * totalArea;
		float* ptr = std::lower_bound(triangleAreas, &(triangleAreas[ntris-1])+1, randomnum);
		int offset = (int) (ptr - triangleAreas);
		neightris[offset].numMoveables++;
		MoveablePoint* newmp = new MoveablePoint();
		float b1 = RandomFloat();
		float b2 = (1.f - b1) * RandomFloat();
		newmp->b1 = b1; newmp->b2 = b2; newmp->triidx = i;
		newmp->next = neightris[offset].mps;
		neightris[offset].mps = newmp;
		PointList* np = new PointList();
		const Point &p1 = p[neightris[offset].v1];
		const Point &p2 = p[neightris[offset].v2];
		const Point &p3 = p[neightris[offset].v3];
		np->p = newmp->b1 * p1 + newmp->b2 * p2 + (1.f - newmp->b1 - newmp->b2) * p3;
		np->next = neightris[offset].plist1;
		neightris[offset].plist1 = np;
	}
	numPoints = numPointsAssigned;
#endif
}
TriangleMesh::~TriangleMesh() {
	delete[] vertexIndex;
	delete[] p;
	delete[] s;
	delete[] n;
	delete[] uvs;
}
#ifdef __CODEDEFT__
void TriangleMesh::LoadFile()
{
	FILE* file = fopen(fileName.c_str(), "rb");
	if (!file) return;
	int count = 0;
	float totalArea = 0.f;
	fread(&count, sizeof(int), 1, file);
	fread(&totalArea, sizeof(float), 1, file);
	for (int i = 0; i < count; i++) {
		Point loadedP;
		fread(&(loadedP.x), sizeof(float), 1, file);
		fread(&(loadedP.y), sizeof(float), 1, file);
		fread(&(loadedP.z), sizeof(float), 1, file);
		loadedPoints.push_back(loadedP);
	}
	fclose(file);
	bFileLoaded = true;
}
void TriangleMesh::saveFlag() const
{
	bSave = true;
}

bool TriangleMesh::CompareEdges(SharedEdge& se1, SharedEdge& se2)
{
	if ((se1.p1 == se2.p1) && (se1.p2 == se2.p2)) {
		return true;
	} else if ((se1.p2 == se2.p1) && (se1.p1 == se2.p2)) {
		return true;
	}

	return false;
}

int TriangleMesh::GetSharedNeighbor(SharedEdge& se)
{
	int key = ((int)abs(floorf(se.p1.x) + floorf(se.p2.x))) % nverts;

	// Search through the table
	if (seHashTable[key].se.tri1Id == -1) {
		seHashTable[key].se = se;
		return -1;
	}

	SEList* prev = 0;
	SEList* cur = &(seHashTable[key]);
	while (cur != 0) {
		if (CompareEdges(cur->se, se)) {
			if ( cur->se.tri1Id == -1 ||
				((cur->se.tri1Id != -1) &&
				 (cur->se.tri2Id != -1)) ) {
				printf("Multiplly shared edges !\n");
				bValidNeighbors = false;
				return -1;
			}
			cur->se.tri2Id = se.tri1Id;
			return cur->se.tri1Id;
		}
		prev = cur;
		cur = cur->next;
	}

	// Need to clean these up
	prev->next = new SEList();
	prev->next->se = se;
	return -1;
}

void TriangleMesh::UpdateNeighbors(int tri1, int tri2,
				   SharedEdge& se)
{
	if (CompareEdges(neightris[tri1].e1, se)) {
		neightris[tri1].e1 = se;
		neightris[tri1].neighbor1 = &neightris[tri2];
	} else if (CompareEdges(neightris[tri1].e2, se)) {
		neightris[tri1].e2 = se;
		neightris[tri1].neighbor2 = &neightris[tri2];
	} else if (CompareEdges(neightris[tri1].e3, se)) {
		neightris[tri1].e3 = se;
		neightris[tri1].neighbor3 = &neightris[tri2];
	}

	if (CompareEdges(neightris[tri2].e1, se)) {
		neightris[tri2].n1 = tri1;
		neightris[tri2].e1 = se;
		neightris[tri2].neighbor1 = &neightris[tri1];
	} else if (CompareEdges(neightris[tri2].e2, se)) {
		neightris[tri2].n2 = tri1;
		neightris[tri2].e2 = se;
		neightris[tri2].neighbor2 = &neightris[tri1];
	} else if (CompareEdges(neightris[tri2].e3, se)) {
		neightris[tri2].n3 = tri1;
		neightris[tri2].e3 = se;
		neightris[tri2].neighbor3 = &neightris[tri1];
	}
}
#endif
BBox TriangleMesh::ObjectBound() const {
	BBox bobj;
	for (int i = 0; i < nverts; i++)
		bobj = Union(bobj, WorldToObject(p[i]));
	return bobj;
}
BBox TriangleMesh::WorldBound() const {
	BBox worldBounds;
	for (int i = 0; i < nverts; i++)
		worldBounds = Union(worldBounds, p[i]);
	return worldBounds;
}

#ifdef __CODEDEFT__
int TriangleMesh::StrapTo(PointList* curpl, int* tri, int numrecurses) const
{
	if (numrecurses > 30) {
		return -1;
	}

	Point p1 = p[neightris[*tri].v1];
	Point p2 = p[neightris[*tri].v2];
	Point p3 = p[neightris[*tri].v3];

	Vector side1crossn = Cross(neightris[*tri].n, Vector(p2-p1));
	Vector side2crossn = Cross(neightris[*tri].n, Vector(p3-p2));
	Vector side3crossn = Cross(neightris[*tri].n, Vector(p1-p3));
	side1crossn = Normalize(side1crossn);
	side2crossn = Normalize(side2crossn);
	side3crossn = Normalize(side3crossn);
	float costheta = 0.f;
	float theta = 0.f;
	float sintheta = 0.f;
	float frontorback = 0.f;
	Vector movetoorigin;
	Vector axis;
	Vector tmp = Vector(curpl->p - p2);
	tmp = Normalize(tmp);
	float Dottmp = Dot(side2crossn, tmp);
	Vector tmp2 = Vector(curpl->p - p3);
	tmp2 = Normalize(tmp2);
	float Dottmp2 = Dot(side3crossn, tmp2);
	Vector tmp3 = Vector(curpl->p - p1);
	tmp3 = Normalize(tmp3);
	float Dottmp3 = Dot(side1crossn, tmp3);

	int prevtri = *tri;

	if (Dottmp < -0.001f) {
		if (neightris[*tri].n2 < 0) {
			// If it's outside the polyhedron, move it back inside the
			// triangle
			Point sp = p[neightris[*tri].v2];
			Vector moveV(curpl->p - sp);
			float coeff = Dot(moveV, side2crossn);
			curpl->p = curpl->p - coeff * side2crossn;
			return StrapTo(curpl, tri, numrecurses+1);
		} else {
			// Outside the second side
			costheta = Dot(neightris[*tri].n, neightris[neightris[*tri].n2].n);
			theta = acosf(costheta);
			if (!(theta >= 0.f)) {
//				printf("%f %f\n", costheta, theta);
				theta = 0.f;
			}
			frontorback = Dot(neightris[neightris[*tri].n2].n, Vector(curpl->p - p2));
			if (frontorback < 0.f) {
				theta = -theta;
			}
			sintheta = sinf(theta);
			axis = Vector(p3 - p2);
			movetoorigin = Vector(p2.x, p2.y, p2.z);
			curpl->p = curpl->p - movetoorigin;
			*tri = neightris[*tri].n2;
		}
	} else if (Dottmp2 < -0.001f) {
		if (neightris[*tri].n3 < 0) {
			// If it's outside the polyhedron, move it back into the
			// triangle
			Point sp = p[neightris[*tri].v3];
			Vector moveV(curpl->p - sp);
			float coeff = Dot(moveV, side3crossn);
			curpl->p = curpl->p - coeff * side3crossn;
			return StrapTo(curpl, tri, numrecurses+1);
		} else {
			// Outside the third side
			costheta = Dot(neightris[*tri].n, neightris[neightris[*tri].n3].n);
			theta = acosf(costheta);
			if (!(theta >= 0.f)) {
//				printf("%f %f\n", costheta, theta);
				theta = 0.f;
			}
			frontorback = Dot(neightris[neightris[*tri].n3].n, Vector(curpl->p - p3));
			if (frontorback < 0.f) {
				theta = -theta;
			}
			sintheta = sinf(theta);
			axis = Vector(p1 - p3);
			movetoorigin = Vector(p3.x, p3.y, p3.z);
			curpl->p = curpl->p - movetoorigin;
			*tri = neightris[*tri].n3;
		}
	} else if (Dottmp3 < -0.001f) {
		if (neightris[*tri].n1 < 0) {
			// If it's outside the polyhedron, move it back into
			// the triangle
			Point sp = p[neightris[*tri].v1];
			Vector moveV(curpl->p - sp);
			float coeff = Dot(moveV, side1crossn);
			curpl->p = curpl->p - coeff * side1crossn;
			return StrapTo(curpl, tri, numrecurses+1);
		} else {
			// Outside the first side
			costheta = Dot(neightris[*tri].n, neightris[neightris[*tri].n1].n);
			theta = acosf(costheta);
			if (!(theta >= 0.f)) {
//				printf("%f %f\n", costheta, theta);
				theta = 0.f;
			}
			frontorback = Dot(neightris[neightris[*tri].n1].n, Vector(curpl->p - p1));
			if (frontorback < 0.f) {
				theta = -theta;
			}
			sintheta = sinf(theta);
			axis = Vector(p2 - p1);
			movetoorigin = Vector(p1.x, p1.y, p1.z);
			curpl->p = curpl->p - movetoorigin;
			*tri = neightris[*tri].n1;
		}
	} else {
		// Inside
		return -1;
	}
	axis = Normalize(axis);
	float mat[4][4];
	mat[0][0] = costheta +(1.f-costheta)*axis.x*axis.x;
	mat[1][0] = (1.f-costheta)*axis.x*axis.y + axis.z*sintheta;
	mat[2][0] = (1.f-costheta)*axis.x*axis.z - axis.y*sintheta;
	mat[3][0] = 0.f;
	mat[0][1] = (1.f-costheta)*axis.x*axis.y - axis.z*sintheta;
	mat[1][1] = costheta + (1.f-costheta)*axis.y*axis.y;
	mat[2][1] = (1.f-costheta)*axis.y*axis.z + axis.x*sintheta;
	mat[3][1] = 0.f;
	mat[0][2] = (1.f-costheta)*axis.x*axis.z + axis.y*sintheta;
	mat[1][2] = (1.f-costheta)*axis.y*axis.z - axis.x*sintheta;
	mat[2][2] = costheta + (1-costheta)*axis.z*axis.z;
	mat[3][2] = 0.f;
	mat[0][3] = 0.f;
	mat[1][3] = 0.f;
	mat[2][3] = 0.f;
	mat[3][3] = 1.f;
	Point tmpp;
	tmpp.x = mat[0][0] * curpl->p.x + mat[0][1] * curpl->p.y + mat[0][2] * curpl->p.z;
	tmpp.y = mat[1][0] * curpl->p.x + mat[1][1] * curpl->p.y + mat[1][2] * curpl->p.z;
	tmpp.z = mat[2][0] * curpl->p.x + mat[2][1] * curpl->p.y + mat[2][2] * curpl->p.z;
	Point prevp = curpl->p;
	curpl->p = tmpp + movetoorigin;

	Point p_ = curpl->p;
	Vector thisnorm;
	// Now, let's make sure that they're in the same plane
	if (Dottmp < -0.001f) {
		thisnorm = Cross(Vector(p_ - p2), Vector(p3 - p2));
	} else if (Dottmp2 < -0.001f) {
		thisnorm = Cross(Vector(p_ - p3), Vector(p1 - p3));
	} else if (Dottmp3 < -0.001f) {
		thisnorm = Cross(Vector(p_ - p1), Vector(p2 - p1));
	} else {
		Assert(0);
	}
	thisnorm = Normalize(thisnorm);
	costheta = Dot(neightris[*tri].n, thisnorm);
	if (!(costheta > 0.95f)) {
		printf("%f\n", costheta);
		printf("%f %f %f\n", neightris[prevtri].n.x, neightris[prevtri].n.y, neightris[prevtri].n.z);
		if (Dottmp < -0.001f) {
			printf("%f %f %f\n", neightris[neightris[prevtri].n2].n.x,
				neightris[neightris[prevtri].n2].n.y,
				neightris[neightris[prevtri].n2].n.z);
		} else if (Dottmp2 < -0.001f) {
			printf("%f %f %f\n", neightris[neightris[prevtri].n3].n.x,
				neightris[neightris[prevtri].n3].n.y,
				neightris[neightris[prevtri].n3].n.z);
		} else if (Dottmp3 < -0.001f) {
			printf("%f %f %f\n", neightris[neightris[prevtri].n1].n.x,
				neightris[neightris[prevtri].n1].n.y,
				neightris[neightris[prevtri].n1].n.z);
		}
//		printf("%f %f %f\n", costheta, theta, sintheta);
//		printf("%f %f %f   %f %f %f   %f %f %f\n",
//			prevp.x, prevp.y, prevp.z,
//			tmpp.x, tmpp.y, tmpp.z,
//			movetoorigin.x, movetoorigin.y, movetoorigin.z);
//		printf("%f %f %f    %f %f %f     %f %f %f     %f %f %f\n",
//			p_.x, p_.y, p_.z,
//			p1.x, p1.y, p1.z,
//			p2.x, p2.y, p2.z,
//			p3.x, p3.y, p3.z);
//		printf("%f %f %f       %f %f %f\n",
//			neightris[*tri].n.x,
//			neightris[*tri].n.y,
//			neightris[*tri].n.z,
//			thisnorm.x, thisnorm.y, thisnorm.z);
	}

	return *tri;
}
#endif
// YG add begin

void
TriangleMesh::renderGL() const
{
#ifndef __CODEDEFT__
	glVertexPointer(3, GL_FLOAT,0, p);

	glEnable(GL_POLYGON_OFFSET_FILL);
	glPolygonOffset(0.0f, 1.0f);
	glColor3f(0, 0, 0);
	glPolygonMode(GL_FRONT,GL_FILL);
	glDrawElements(GL_TRIANGLES, 3*ntris, GL_UNSIGNED_INT, vertexIndex);

	glDisable(GL_POLYGON_OFFSET_FILL);
	glColor3f(1, 1, 1);
	glPolygonMode(GL_FRONT,GL_LINE);
	glDrawElements(GL_TRIANGLES, 3*ntris, GL_UNSIGNED_INT, vertexIndex);
#else
	/*
	vector<Point>::const_iterator fit = fixedPoints.begin();
	glPointSize(1);
	glBegin(GL_POINTS);
	while (fit != fixedPoints.end()) {
		Point curp = *fit;
		glColor3f(1.f, 1.f, 1.f);
		glVertex3f(curp.x, curp.y, curp.z);
		fit++;
	}
	glEnd();
	glFlush();
	if (bSave) {
		bSave = false;
		FILE* file = fopen(fileName.c_str(), "wb");
		int count = fixedPoints.size();
		fwrite(&count, sizeof(int), 1, file);
		fwrite(&fixedArea, sizeof(float), 1, file);
		vector<Point>::const_iterator cfit = fixedPoints.begin();
		vector<Normal>::const_iterator cnit = fixedNormals.begin();
		vector<float>::const_iterator cait = fixedAreas.begin();
		while (cfit != fixedPoints.end()) {
			Point savep = *cfit;
			Normal trin = *cnit;
			float area = *cait;
			fwrite(&(savep.x), sizeof(float), 1, file);
			fwrite(&(savep.y), sizeof(float), 1, file);
			fwrite(&(savep.z), sizeof(float), 1, file);
			fwrite(&(trin.x), sizeof(float), 1, file);
			fwrite(&(trin.y), sizeof(float), 1, file);
			fwrite(&(trin.z), sizeof(float), 1, file);
			fwrite(&area, sizeof(float), 1, file);
			cfit++;
			cnit++;
			cait++;
		}
		fclose(file);
	}
	return;
	*/





	if (bFileLoaded) {
		// Just render the loaded points
		vector<Point>::const_iterator it = loadedPoints.begin();
		glPointSize(2);
		glBegin(GL_POINTS);
		while (it != loadedPoints.end()) {
			Point curp = *it;
			glColor3f(1.f, 1.f, 1.f);
			glVertex3f(curp.x, curp.y, curp.z);
			it++;
		}
		glEnd();
		glFlush();
		return;
	}

	// Perform an iteration of point repulsion
	bool bPerformRepulsion = true;
	int numgoovers = 2;
	// Impact radius
	float impactradius = 2.f * sqrtf(triangleAreas[ntris-1]/(float)numPoints);
	if (bPerformRepulsion) {
		for (int i = 0; i < ntris; i++) {
			// Go through all the neighbors and
			// store'em.
			NeighSet neighbors;
			IntList* prevTraversed = new IntList();
			prevTraversed->i = i;
			prevTraversed->next = 0;
			int n1 = neightris[i].n1;
			int n2 = neightris[i].n2;
			int n3 = neightris[i].n3;
			if (n1 != -1)
				neightris[n1].TraverseNeighbors(neighbors, numgoovers, &prevTraversed, 1);
			if (n2 != -1)
				neightris[n2].TraverseNeighbors(neighbors, numgoovers, &prevTraversed, 2);
			if (n3 != -1)
				neightris[n3].TraverseNeighbors(neighbors, numgoovers, &prevTraversed, 3);
			delete prevTraversed;

			// Let me compute the neighbor force vector per point

			// Let's rotate all these points
			std::vector<NeighborPoint>::iterator chit = neighbors.neighbors.begin();
			while (chit != neighbors.neighbors.end()) {
				Point p_ = (*chit).p;
				int side = (*chit).edge;
				Point p1 = p[neightris[i].v1];
				Point p2 = p[neightris[i].v2];
				Point p3 = p[neightris[i].v3];
				float frontorback;
				Vector thisnorm;
				if (side == 1) {
					thisnorm = Cross(Vector(p_ - p1), Vector(p2 - p1));
					if (thisnorm.LengthSquared() == 0.f) {
						thisnorm = neightris[i].n;
					}
					frontorback = Dot(neightris[i].n, Vector(p_ - p1));
				} else if (side == 2) {
					thisnorm = Cross(Vector(p_ - p2), Vector(p3 - p2));
					if (thisnorm.LengthSquared() == 0.f) {
						thisnorm = neightris[i].n;
					}
					frontorback = Dot(neightris[i].n, Vector(p_ - p2));
				} else if (side == 3) {
					thisnorm = Cross(Vector(p_ - p3), Vector(p1 - p3));
					if (thisnorm.LengthSquared() == 0.f) {
						thisnorm = neightris[i].n;
					}
					frontorback = Dot(neightris[i].n, Vector(p_ - p3));
				} else {
					Assert(0);
				}
				thisnorm = Normalize(thisnorm);
				float costheta = Dot(neightris[i].n, thisnorm);
				float theta = acosf(costheta);
				if (frontorback < 0.f) {
					// If front, rotate in minus direction
					theta = -theta;
				}
				float sintheta = sinf(theta);
				// Create transformation matrix
				Vector movetoorigin;
				Vector axis;
				if (side == 1) {
					movetoorigin = Vector(p1);
					axis = Vector(p2 - p1);
					Vector tmp = p_ - p1;
					p_ = Point(tmp.x, tmp.y, tmp.z);
				} else if (side == 2) {
					movetoorigin = Vector(p2);
					axis = Vector(p3 - p2);
					Vector tmp = p_ - p2;
					p_ = Point(tmp.x, tmp.y, tmp.z);
				} else if (side == 3) {
					movetoorigin = Vector(p3);
					axis = Vector(p1 - p3);
					Vector tmp = p_ - p3;
					p_ = Point(tmp.x, tmp.y, tmp.z);
				}
				axis = Normalize(axis);
				// Rotate theta degrees around axis
				float mat[4][4];
				mat[0][0] = costheta +(1.f-costheta)*axis.x*axis.x;
				mat[1][0] = (1.f-costheta)*axis.x*axis.y + axis.z*sintheta;
				mat[2][0] = (1.f-costheta)*axis.x*axis.z - axis.y*sintheta;
				mat[3][0] = 0.f;
				mat[0][1] = (1.f-costheta)*axis.x*axis.y - axis.z*sintheta;
				mat[1][1] = costheta + (1.f-costheta)*axis.y*axis.y;
				mat[2][1] = (1.f-costheta)*axis.y*axis.z + axis.x*sintheta;
				mat[3][1] = 0.f;
				mat[0][2] = (1.f-costheta)*axis.x*axis.z + axis.y*sintheta;
				mat[1][2] = (1.f-costheta)*axis.y*axis.z - axis.x*sintheta;
				mat[2][2] = costheta + (1-costheta)*axis.z*axis.z;
				mat[3][2] = 0.f;
				mat[0][3] = 0.f;
				mat[1][3] = 0.f;
				mat[2][3] = 0.f;
				mat[3][3] = 1.f;
				Point tmpp;
				tmpp.x = mat[0][0] * p_.x + mat[0][1] * p_.y + mat[0][2] * p_.z;
				tmpp.y = mat[1][0] * p_.x + mat[1][1] * p_.y + mat[1][2] * p_.z;
				tmpp.z = mat[2][0] * p_.x + mat[2][1] * p_.y + mat[2][2] * p_.z;
				(*chit).p = tmpp + movetoorigin;

				p_ = (*chit).p;
				// Now, let's make sure that they're in the same plane
//				if (side == 1) {
//					thisnorm = Cross(Vector(p_ - p1), Vector(p2 - p1));
//				} else if (side == 2) {
//					thisnorm = Cross(Vector(p_ - p2), Vector(p3 - p2));
//				} else if (side == 3) {
//					thisnorm = Cross(Vector(p_ - p3), Vector(p1 - p3));
//				} else {
//					Assert(0);
//				}
//				thisnorm = Normalize(thisnorm);
//				costheta = Dot(neightris[i].n, thisnorm);
//				printf("%f\n", costheta);

				chit++;
			}

			/*
			if (i == 1) {
				// Let's render the aligned triangle
				glPointSize(5);
				//		glLineWidth(5);
				glBegin(GL_POINTS);
				PointList* pRenderList = 0;
				if (neightris[i].plist1) {
					pRenderList = neightris[i].plist1;
				} else {
					pRenderList = neightris[i].plist2;
				}
				PointList* curRpl = pRenderList;
				while (curRpl) {
					Point curp = curRpl->p;
					float c1 = ((float)((int)((i*50 + 1)) % nverts)) / (float)nverts;
					float c2 = ((float)((int)((i*50 + 2007))% nverts)) / (float)nverts;
					float c3 = ((float)((int)((i*50 + 3007))% nverts)) / (float)nverts;
					glColor3f(c1, c2, c3);
					glVertex3f(curp.x, curp.y, curp.z);
					curRpl = curRpl->next;
				}

				// Now render the rotated neighbor points
				std::vector<NeighborPoint>::const_iterator cit = neighbors.neighbors.begin();
				while (cit != neighbors.neighbors.end()) {
					Point p_ = (*cit).p;
					float c1 = ((float)((int)((i*50 + 1 + 50)) % nverts)) / (float)nverts;
					float c2 = ((float)((int)((i*50 + 2007 + 50))% nverts)) / (float)nverts;
					float c3 = ((float)((int)((i*50 + 3007 + 50))% nverts)) / (float)nverts;
					glColor3f(c1, c2, c3);
					glVertex3f(p_.x, p_.y, p_.z);
					cit++;
				}
				glEnd();
				glFlush();
			}
			*/

			PointList* plist = 0;
			if (neightris[i].plist1) {
				plist = neightris[i].plist1;
			} else {
				plist = neightris[i].plist2;
			}
			PointList* curpl = plist;
			while (curpl) {
				Point curp = curpl->p;
				// Look through the current triangle and
				// compute the forces
				PointList* interpl = plist;
				while (interpl) {
					if (interpl != curpl) {
						Point interp = interpl->p;
						Vector force(curp - interp);
						if (force.Length() < impactradius) {
							curpl->repulsiveForce += Normalize(force) / (force.Length() + 1.f);
						}
					}
					interpl = interpl->next;
				}

				// Now let's look at all those rotated points
				std::vector<NeighborPoint>::const_iterator cit = neighbors.neighbors.begin();
				while (cit != neighbors.neighbors.end()) {
					Point p_ = (*cit).p;

					// Catch drifters
					Point p1 = p[neightris[i].v1];
					Point p2 = p[neightris[i].v2];
					Vector candidate1(p_ - p1);
					Vector candidate2(p_ - p2);
					float drift = 0.f;
					if (candidate1.LengthSquared() > candidate2.LengthSquared()) {
						drift = Dot(candidate1, neightris[i].n);
					} else {
						drift = Dot(candidate2, neightris[i].n);
					}
					p_ -= drift * neightris[i].n;

					Vector force(curp - p_);
					if (force.Length() < impactradius) {
						curpl->repulsiveForce += Normalize(force) / (force.Length() + 1.f);
					}
					cit++;
				}
				curpl = curpl->next;
			}

			/*
			std::set<NeighborPoint>::const_iterator it;
			it = neighbors.begin();
			while (it != neighbors.end()) {
			NeighborPoint* np = *it;
			Point p = np->p;
			Point newP;
			if (np->edge == 1) {
			newP = neightris[i].RotateAround1(p);
			} else if (np->edge == 2) {
			newP = neightris[i].RotateAround2(p);
			} else if (np->edge == 3) {
			newP = neightris[i].RotateAround3(p);
			}
			MoveablePoint* nmp = neightris[i].mps;
			for (int k = 0; k < neightris[i].numMoveables; k++) {
			Point po = nmp->b1 * p[neightris[i].v1] + nmp->b2 * p[neightris[i].v2] +
			(1.f - nmp->b1 - nmp->b2) * p[neightris[i].v3];
			// This point's influence
			Vector vec(p - newP);
			float dist = vec.Length();
			if (dist < impactradius) {
			nmp->repulsiveForce += (1.f/dist)*vec;
			}
			nmp = nmp->next;
			}
			it++;
			}
			*/
		}

		/*
		for (int i = 0; i < ntris; i++) {
		MoveablePoint* nmp = neightris[i].mps;
		for (int k = 0; k < neightris[i].numMoveables; k++) {
		nmp->repulsiveForce -= neightris[i].normal;
		nmp->Move(nmp->repulsiveForce);
		nmp->Neighbor();
		nmp->Project();
		nmp = nmp->next;
		}
		}
		*/

		// Now, let's move points
		// The list to move the points to
		static int curlist = 2;

		for (int i = 0; i < ntris; i++) {
			NeighborTri* neighbor = &neightris[i];
			PointList* plist = 0;
			if (curlist == 2) {
				plist = neighbor->plist1;
				neighbor->plist1 = 0;
			} else if (curlist == 1) {
				plist = neighbor->plist2;
				neighbor->plist2 = 0;
			}

			PointList* curpl = plist;
			while (curpl) {
//				printf("Hello\n");
//				printf("%f %f %f\n", curpl->p.x, curpl->p.y, curpl->p.z);
				curpl->p = curpl->p + FORCE_COEFF * curpl->repulsiveForce;
//				printf("%f %f %f\n", curpl->p.x, curpl->p.y, curpl->p.z);

				Point p1 = p[neightris[i].v1];
				Point p2 = p[neightris[i].v2];
				Vector candidate1(curpl->p - p1);
				Vector candidate2(curpl->p - p2);
				float drift = 0.f;
				if (candidate1.LengthSquared() > candidate2.LengthSquared()) {
					drift = Dot(candidate1, neightris[i].n);
				} else {
					drift = Dot(candidate2, neightris[i].n);
				}
				curpl->p -= drift * neightris[i].n;

				PointList* nextpl = curpl->next;
				int side = i;
				int r;
				while ((r = StrapTo(curpl, &side, 1)) != -1);
				curpl->repulsiveForce = Vector(0.f, 0.f, 0.f);

				// Make sure that the points remain tightly on the surface
//				drift = Dot(Vector(curpl->p - p1), neightris[side].n);
//				curpl->p -= drift * neightris[side].n;

				if (curlist == 1) {
					curpl->next = neightris[side].plist1;
					neightris[side].plist1 = curpl;
				} else if (curlist == 2) {
					curpl->next = neightris[side].plist2;
					neightris[side].plist2 = curpl;
				} else {
					Assert(0);
				}
				curpl = nextpl;
			}
		}

		// Switch list for the next time
		if (curlist == 1) curlist = 2;
		else curlist = 1;

		// Check that things are being converged to uniform point
		// spacing
		int count = 0;
//		float c1 = 1.f, c2 = 1.f, c3 = 1.f;
//		glColor3f(c1, c2, c3);
		glPointSize(2);
//		glLineWidth(5);
		glBegin(GL_POINTS);
		for (int i = 0; i < ntris; i++) {
			PointList* curpl = 0;
			if (curlist == 1)
				curpl = neightris[i].plist2;
			else
				curpl = neightris[i].plist1;
			while (curpl) {
				Point curp = curpl->p;
				float c1 = ((float)((int)((i*50 + 1)) % nverts)) / (float)nverts;
				float c2 = ((float)((int)((i*50 + 2007))% nverts)) / (float)nverts;
				float c3 = ((float)((int)((i*50 + 3007))% nverts)) / (float)nverts;
				glColor3f(c1, c2, c3);
				glVertex3f(curp.x, curp.y, curp.z);
//				glVertex3f(curp.x+1.f, curp.y+1.f, curp.z+1.f);
				count++;
//				glVertex3f(curp.x + 200.f*neightris[i].n.x, curp.y + 200.f*neightris[i].n.y, curp.z + 200.f*neightris[i].n.z);
				curpl = curpl->next;
			}
		}
		glEnd();
		glFlush();
//		printf("%d\n", count);

		if (bSave) {
			bSave = false;
			FILE* file = fopen(fileName.c_str(), "wb");
			fwrite(&count, sizeof(int), 1, file);
			fwrite(&(triangleAreas[ntris-1]), sizeof(float), 1, file);
			for (int i = 0; i < ntris; i++) {
				PointList* curpl = 0;
				if (curlist == 1)
					curpl = neightris[i].plist2;
				else
					curpl = neightris[i].plist1;
				while (curpl) {
					Point savep = curpl->p;
					savep = WorldToObject(savep);
					Normal trin(neightris[i].n);
					trin = WorldToObject(trin);
					fwrite(&(savep.x), sizeof(float), 1, file);
					fwrite(&(savep.y), sizeof(float), 1, file);
					fwrite(&(savep.z), sizeof(float), 1, file);
					fwrite(&(trin.x), sizeof(float), 1, file);
					fwrite(&(trin.y), sizeof(float), 1, file);
					fwrite(&(trin.z), sizeof(float), 1, file);
					curpl = curpl->next;
				}
			}
			fclose(file);
		}

		/*
			while (curpl) {
				glPointSize(5);
				glBegin(GL_POINTS);
				glColor3f(c1, c2, c3);
				glVertex3f(curpl->p.x, curpl->p.y, curpl->p.z);
				glEnd();
				curpl = curpl->next;
			}
			std::vector<NeighborPoint>::const_iterator cit = neighbors.neighbors.begin();
			while (cit != neighbors.neighbors.end()) {
				Point p_ = (*cit).p;
				Vector distvec(p_ - curp);
				if (distvec.Length() < impactradius) {
					glPointSize(5);
					glBegin(GL_POINTS);
					glColor3f(c1, c2, c3);
					glVertex3f(p_.x, p_.y, p_.z);
					glEnd();
				}
				cit++;
			}
		}
		*/
	}

	/*
	if (bValidNeighbors) {
		// Draw neighbor triangles visibly
		for (int i = 0; i < ntris; i++) {
			int v1 = vertexIndex[3*i];
			int v2 = vertexIndex[3*i+1];
			int v3 = vertexIndex[3*i+2];

			Point &p1 = p[v1];
			Point &p2 = p[v2];
			Point &p3 = p[v3];

			glPolygonMode(GL_FRONT,GL_FILL);
			neightris[i].RenderGL(p1, p2, p3, nverts);
		}
	}
	*/
#endif
}

// YG add end

void
TriangleMesh::Refine(vector<Reference<Shape> > &refined)
const {
	for (int i = 0; i < ntris; ++i) {
		refined.push_back(new Triangle(ObjectToWorld,
		                               reverseOrientation,
                                       (TriangleMesh *)this,
									   i));
	}
}

Triangle::Triangle(const Transform &o2w, bool ro,
	         TriangleMesh *m, int n)
			: Shape(o2w, ro) {
		mesh = m;
		v = &mesh->vertexIndex[3*n];
		faceIndex = n;		// YG add
		// Update created triangles stats
		static StatsCounter trisMade("Geometry",
		                             "Triangles created");
		++trisMade;
}

BBox Triangle::ObjectBound() const {
	// Get triangle vertices in _p1_, _p2_, and _p3_
	const Point &p1 = mesh->p[v[0]];
	const Point &p2 = mesh->p[v[1]];
	const Point &p3 = mesh->p[v[2]];
	return Union(BBox(WorldToObject(p1), WorldToObject(p2)),
		WorldToObject(p3));
}

BBox Triangle::WorldBound() const {
	// Get triangle vertices in _p1_, _p2_, and _p3_
	const Point &p1 = mesh->p[v[0]];
	const Point &p2 = mesh->p[v[1]];
	const Point &p3 = mesh->p[v[2]];
	return Union(BBox(p1, p2), p3);
}

bool Triangle::Intersect(const Ray &ray, float *tHit,
		DifferentialGeometry *dg) const {
	// Initialize triangle intersection statistics
	static
	StatsPercentage triangleHits("Geometry",
	                             "Triangle Ray Intersections");
	// Update triangle tests count
	triangleHits.Add(0, 1);
	// Compute $\VEC{s}_1$
	// Get triangle vertices in _p1_, _p2_, and _p3_
	const Point &p1 = mesh->p[v[0]];
	const Point &p2 = mesh->p[v[1]];
	const Point &p3 = mesh->p[v[2]];
	Vector e1 = p2 - p1;
	Vector e2 = p3 - p1;
	Vector s1 = Cross(ray.d, e2);
	float divisor = Dot(s1, e1);
	if (divisor == 0.)
		return false;
	float invDivisor = 1.f / divisor;
	// Compute first barycentric coordinate
	Vector d = ray.o - p1;
	float b1 = Dot(d, s1) * invDivisor;
	if (b1 < 0. || b1 > 1.)
		return false;
	// Compute second barycentric coordinate
	Vector s2 = Cross(d, e1);
	float b2 = Dot(ray.d, s2) * invDivisor;
	if (b2 < 0. || b1 + b2 > 1.)
		return false;
	// Compute _t_ to intersection point
	float t = Dot(e2, s2) * invDivisor;
	if (t < ray.mint || t > ray.maxt)
		return false;
	triangleHits.Add(1, 0); //NOBOOK
	// Fill in _DifferentialGeometry_ from triangle hit
	// Compute triangle partial derivatives
	Vector dpdu, dpdv;
	float uvs[3][2];
	GetUVs(uvs);
	// Compute deltas for triangle partial derivatives
	float du1 = uvs[0][0] - uvs[2][0];
	float du2 = uvs[1][0] - uvs[2][0];
	float dv1 = uvs[0][1] - uvs[2][1];
	float dv2 = uvs[1][1] - uvs[2][1];
	Vector dp1 = p1 - p3, dp2 = p2 - p3;
	float determinant = du1 * dv2 - dv1 * du2;
	if (determinant == 0.f) {
		// Handle zero determinant for triangle partial derivative matrix
		CoordinateSystem(Normalize(Cross(e2, e1)), &dpdu, &dpdv);
	}
	else {
		float invdet = 1.f / determinant;
		dpdu = ( dv2 * dp1 - dv1 * dp2) * invdet;
		dpdv = (-du2 * dp1 + du1 * dp2) * invdet;
	}
	// Interpolate $(u,v)$ triangle parametric coordinates
	float b0 = 1 - b1 - b2;
	float tu = b0*uvs[0][0] + b1*uvs[1][0] + b2*uvs[2][0];
	float tv = b0*uvs[0][1] + b1*uvs[1][1] + b2*uvs[2][1];
	*dg = DifferentialGeometry(ray(t), dpdu, dpdv,
	                           Vector(0,0,0), Vector(0,0,0),
							   tu, tv, this);
	*tHit = t;
	return true;
}

bool Triangle::IntersectP(const Ray &ray) const {
	// Initialize triangle intersection statistics
	static
	StatsPercentage triangleHits("Geometry",
	                             "Triangle Ray Intersections");
	// Update triangle tests count
	triangleHits.Add(0, 1);
	// Compute $\VEC{s}_1$
	// Get triangle vertices in _p1_, _p2_, and _p3_
	const Point &p1 = mesh->p[v[0]];
	const Point &p2 = mesh->p[v[1]];
	const Point &p3 = mesh->p[v[2]];
	Vector e1 = p2 - p1;
	Vector e2 = p3 - p1;
	Vector s1 = Cross(ray.d, e2);
	float divisor = Dot(s1, e1);
	if (divisor == 0.)
		return false;
	float invDivisor = 1.f / divisor;
	// Compute first barycentric coordinate
	Vector d = ray.o - p1;
	float b1 = Dot(d, s1) * invDivisor;
	if (b1 < 0. || b1 > 1.)
		return false;
	// Compute second barycentric coordinate
	Vector s2 = Cross(d, e1);
	float b2 = Dot(ray.d, s2) * invDivisor;
	if (b2 < 0. || b1 + b2 > 1.)
		return false;
	// Compute _t_ to intersection point
	float t = Dot(e2, s2) * invDivisor;
	if (t < ray.mint || t > ray.maxt)
		return false;
	triangleHits.Add(1, 0); //NOBOOK
	return true;
}
void Triangle::GetUVs(float uv[3][2]) const {
	if (mesh->uvs) {
		uv[0][0] = mesh->uvs[2*v[0]];
		uv[0][1] = mesh->uvs[2*v[0]+1];
		uv[1][0] = mesh->uvs[2*v[1]];
		uv[1][1] = mesh->uvs[2*v[1]+1];
		uv[2][0] = mesh->uvs[2*v[2]];
		uv[2][1] = mesh->uvs[2*v[2]+1];
	} else {
		uv[0][0] = 0.; uv[0][1] = 0.;
		uv[1][0] = 1.; uv[1][1] = 0.;
		uv[2][0] = 1.; uv[2][1] = 1.;
	}
}
float Triangle::Area() const {
	// Get triangle vertices in _p1_, _p2_, and _p3_
	const Point &p1 = mesh->p[v[0]];
	const Point &p2 = mesh->p[v[1]];
	const Point &p3 = mesh->p[v[2]];
	return 0.5f * Cross(p2-p1, p3-p1).Length();
}
Point Triangle::Sample(float u1, float u2,
		Normal *Ns) const {
	float b1, b2;
	UniformSampleTriangle(u1, u2, &b1, &b2);
	// Get triangle vertices in _p1_, _p2_, and _p3_
	const Point &p1 = mesh->p[v[0]];
	const Point &p2 = mesh->p[v[1]];
	const Point &p3 = mesh->p[v[2]];
	Point p = b1 * p1 + b2 * p2 + (1.f - b1 - b2) * p3;
	Normal n = Normal(Cross(p2-p1, p3-p1));
	*Ns = Normalize(n);
	if (reverseOrientation) *Ns *= -1.f;
	return p;
}

void 
Triangle::GetShadingGeometry(const Transform &obj2world,
		const DifferentialGeometry &dg,
		DifferentialGeometry *dgShading) const {
	if (!mesh->n && !mesh->s) {
		*dgShading = dg;
		return;
	}
	// Initialize _Triangle_ shading geometry with _n_ and _s_
	// Compute barycentric coordinates for point
	float b[3];
	// Initialize _A_ and _C_ matrices for barycentrics
	float uv[3][2];
	GetUVs(uv);
	float A[2][2] =
		{ { uv[1][0] - uv[0][0], uv[2][0] - uv[0][0] },
		    { uv[1][1] - uv[0][1], uv[2][1] - uv[0][1] } };
	float C[2] = { dg.u - uv[0][0], dg.v - uv[0][1] };
	if (!SolveLinearSystem2x2(A, C, &b[1])) {
		// Handle degenerate parametric mapping
		b[0] = b[1] = b[2] = 1.f/3.f;
	}
	else
		b[0] = 1.f - b[1] - b[2];
	// Use _n_ and _s_ to compute shading tangents for triangle, _ss_ and _ts_
	Normal ns;
	Vector ss, ts;
	if (mesh->n) ns = Normalize(obj2world(b[0] * mesh->n[v[0]] +
		b[1] * mesh->n[v[1]] + b[2] * mesh->n[v[2]]));
	else   ns = dg.nn;
	if (mesh->s) ss = Normalize(obj2world(b[0] * mesh->s[v[0]] +
		b[1] * mesh->s[v[1]] + b[2] * mesh->s[v[2]]));
	else   ss = Normalize(dg.dpdu);
	ts = Normalize(Cross(ss, ns));
	ss = Cross(ts, ns);
	Vector dndu, dndv;
	// Compute \dndu and \dndv for triangle shading geometry
	float uvs[3][2];
	GetUVs(uvs);
	// Compute deltas for triangle partial derivatives of normal
	float du1 = uvs[0][0] - uvs[2][0];
	float du2 = uvs[1][0] - uvs[2][0];
	float dv1 = uvs[0][1] - uvs[2][1];
	float dv2 = uvs[1][1] - uvs[2][1];
	Vector dn1 = Vector(mesh->n[v[0]] - mesh->n[v[2]]);
	Vector dn2 = Vector(mesh->n[v[1]] - mesh->n[v[2]]);
	float determinant = du1 * dv2 - dv1 * du2;
	if (determinant == 0)
		dndu = dndv = Vector(0,0,0);
	else {
		float invdet = 1.f / determinant;
		dndu = ( dv2 * dn1 - dv1 * dn2) * invdet;
		dndv = (-du2 * dn1 + du1 * dn2) * invdet;
	}
	*dgShading = DifferentialGeometry(dg.p, ss, ts,
		dndu, dndv, dg.u, dg.v, dg.shape);
	dgShading->dudx = dg.dudx;  dgShading->dvdx = dg.dvdx; // NOBOOK
	dgShading->dudy = dg.dudy;  dgShading->dvdy = dg.dvdy; // NOBOOK
	dgShading->dpdx = dg.dpdx;  dgShading->dpdy = dg.dpdy; // NOBOOK
}

extern "C" DLLEXPORT Shape *CreateShape(const Transform &o2w,
		bool reverseOrientation, const ParamSet &params) {
	int nvi, npi, nuvi, nsi, nni;
#ifdef __CODEDEFT__
	string fileName = params.FindOneString("filename", "points.dat");
#endif
	const int *vi = params.FindInt("indices", &nvi);
	const Point *P = params.FindPoint("P", &npi);
	const float *uvs = params.FindFloat("uv", &nuvi);
	if (!uvs) uvs = params.FindFloat("st", &nuvi);
	// XXX should complain if uvs aren't an array of 2...
	if (!vi || !P) return NULL;
	const Vector *S = params.FindVector("S", &nsi);
	if (S && nsi != npi) {
		Error("Number of \"S\"s for triangle mesh must match \"P\"s");
		S = NULL;
	}
	const Normal *N = params.FindNormal("N", &nni);
	if (N && nni != npi) {
		Error("Number of \"N\"s for triangle mesh must match \"P\"s");
		N = NULL;
	}
	if (uvs && N) {
		// if there are normals, check for bad uv's that
		// give degenerate mappings; discard them if so
		const int *vp = vi;
		for (int i = 0; i < nvi; i += 3, vp += 3) {
			float area = .5f * Cross(P[vp[0]]-P[vp[1]], P[vp[2]]-P[vp[1]]).Length();
			if (area < 1e-7) continue; // ignore degenerate tris.
			if ((uvs[2*vp[0]] == uvs[2*vp[1]] &&
				uvs[2*vp[0]+1] == uvs[2*vp[1]+1]) ||
				(uvs[2*vp[1]] == uvs[2*vp[2]] &&
				uvs[2*vp[1]+1] == uvs[2*vp[2]+1]) ||
				(uvs[2*vp[2]] == uvs[2*vp[0]] &&
				uvs[2*vp[2]+1] == uvs[2*vp[0]+1])) {
				Warning("Degenerate uv coordinates in triangle mesh.  Discarding all uvs.");
				uvs = NULL;
				break;
			}
		}
	}
	for (int i = 0; i < nvi; ++i)
		if (vi[i] >= npi) {
			Error("trianglemesh has out of-bounds vertex index %d (%d \"P\" values were given",
				vi[i], npi);
			return NULL;
		}

	// YG add begin, remove degenerate triangle
	for (int i = 0; i < nvi; i += 3) {
		Vector normal = Cross(P[vi[i+2]] - P[vi[i+0]], P[vi[i+1]] - P[vi[i+0]]);
		if ( normal.Length() == 0) {
			memmove((int*)vi+i, vi+i+3, (nvi - i - 3)*sizeof(int));
			i -= 3;
			nvi -= 3;
		}
	}

	// YG add end
#ifdef __CODEDEFT__
	TriangleMesh* r = new TriangleMesh(o2w, reverseOrientation, nvi/3, npi, vi, P,
		N, S, uvs);
	r->fileName = fileName;
//	r->LoadFile();
	return r;
#else	
	return new TriangleMesh(o2w, reverseOrientation, nvi/3, npi, vi, P,
		N, S, uvs);
#endif
}
